{{-- resources/views/laptops/index.blade.php --}}
@extends('layouts.base')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
    <div class="bg-white overflow-hidden shadow-sm sm:rounded-lg">
        <div class="p-4 md:p-6 bg-white border-b border-gray-200">
            <!-- Header -->
            <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center mb-6 gap-4">
                <h2 class="text-xl md:text-2xl font-bold text-gray-900">Laptops Management</h2>
                <div class="flex flex-col sm:flex-row gap-2 sm:gap-4">
                    <button onclick="downloadSelectedBarcodes()"
                            id="bulkDownloadBtn"
                            class="bg-green-600 hover:bg-green-700 text-white px-3 md:px-4 py-2 rounded-md text-sm hidden">
                        Download Selected Barcodes
                    </button>
                    <button onclick="document.getElementById('addLaptopModal').classList.remove('hidden')"
                            class="bg-blue-600 hover:bg-blue-700 text-white px-3 md:px-4 py-2 rounded-md text-sm">
                        Add New Laptop
                    </button>
                </div>
            </div>

            <!-- Filters -->
            <div class="mb-6">
                <form method="GET" class="flex flex-col sm:flex-row flex-wrap gap-3 sm:gap-4">
                    <select name="class_id" class="border-gray-300 rounded-md shadow-sm text-sm">
                        <option value="">All Classes</option>
                        @foreach($classes as $class)
                            <option value="{{ $class->id }}" {{ request('class_id') == $class->id ? 'selected' : '' }}>
                                {{ $class->name }}
                            </option>
                        @endforeach
                    </select>

                    <select name="status" class="border-gray-300 rounded-md shadow-sm text-sm">
                        <option value="">All Statuses</option>
                        <option value="available" {{ request('status') == 'available' ? 'selected' : '' }}>Available</option>
                        <option value="checked_out" {{ request('status') == 'checked_out' ? 'selected' : '' }}>Checked Out</option>
                        <option value="maintenance" {{ request('status') == 'maintenance' ? 'selected' : '' }}>Maintenance</option>
                        <option value="damaged" {{ request('status') == 'damaged' ? 'selected' : '' }}>Damaged</option>
                    </select>

                    <button type="submit" class="bg-gray-600 hover:bg-gray-700 text-white px-3 md:px-4 py-2 rounded-md text-sm sm:w-auto w-full">
                        Filter
                    </button>
                </form>
            </div>

            <!-- Laptops Table -->
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                                <input type="checkbox" id="selectAll" class="rounded border-gray-300">
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Asset Tag</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Brand/Model</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Class</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Assigned Student</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Current User</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        @forelse($laptops as $laptop)
                        <tr>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <input type="checkbox" class="laptop-checkbox rounded border-gray-300" value="{{ $laptop->id }}">
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                {{ $laptop->asset_tag }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                {{ $laptop->brand }} {{ $laptop->model }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                {{ $laptop->class->name }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                @if($laptop->assigned_student_name)
                                    <div>
                                        <div class="font-medium">{{ $laptop->assigned_student_name }}</div>
                                        @if($laptop->assigned_student_id)
                                            <div class="text-xs text-gray-400">ID: {{ $laptop->assigned_student_id }}</div>
                                        @endif
                                    </div>
                                @else
                                    <span class="text-gray-400">-</span>
                                @endif
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full
                                    {{ $laptop->status === 'available' ? 'bg-green-100 text-green-800' : '' }}
                                    {{ $laptop->status === 'checked_out' ? 'bg-yellow-100 text-yellow-800' : '' }}
                                    {{ $laptop->status === 'maintenance' ? 'bg-red-100 text-red-800' : '' }}
                                    {{ $laptop->status === 'damaged' ? 'bg-red-100 text-red-800' : '' }}">
                                    {{ ucfirst($laptop->status) }}
                                </span>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                @if($laptop->currentAssignment)
                                    {{ $laptop->currentAssignment->student->full_name }}
                                @else
                                    -
                                @endif
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                <a href="{{ route('laptops.show', $laptop) }}"
                                   class="text-blue-600 hover:text-blue-900 mr-3">View</a>
                                <a href="{{ route('laptops.barcode-pdf', $laptop) }}"
                                   class="text-green-600 hover:text-green-900">Barcode</a>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="8" class="px-6 py-4 text-center text-gray-500">
                                No laptops found.
                            </td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <div class="mt-6">
                {{ $laptops->links() }}
            </div>
        </div>
    </div>
</div>

<!-- Add Laptop Modal -->
<div id="addLaptopModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white rounded-lg max-w-4xl w-full p-8 max-h-[95vh] overflow-y-auto">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-semibold">Add New Laptop</h3>
                <button onclick="document.getElementById('addLaptopModal').classList.add('hidden')"
                        class="text-gray-400 hover:text-gray-600">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>

            <form action="{{ route('laptops.store') }}" method="POST">
                @csrf
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Asset Tag <span class="text-gray-500 text-xs">(Auto-generated if empty)</span></label>
                        <input type="text" name="asset_tag" placeholder="Leave empty for auto-generation" class="w-full border-gray-300 rounded-md shadow-sm">
                        <p class="text-xs text-gray-500 mt-1">Format: LT-2025-001 (Year-Sequential)</p>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Brand</label>
                        <input type="text" name="brand" required class="w-full border-gray-300 rounded-md shadow-sm">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Model</label>
                        <input type="text" name="model" required class="w-full border-gray-300 rounded-md shadow-sm">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Serial Number</label>
                        <input type="text" name="serial_number" required class="w-full border-gray-300 rounded-md shadow-sm">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Class</label>
                        <select name="class_id" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                            @foreach($classes as $class)
                                <option value="{{ $class->id }}">{{ $class->name }}</option>
                            @endforeach
                        </select>
                    </div>

                    <!-- Student Assignment Section -->
                    <div class="border-t border-gray-200 pt-4 mt-4">
                        <h4 class="text-sm font-semibold text-gray-800 mb-3">Student Assignment (Optional)</h4>

                        <div class="space-y-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Search Student</label>

                                <!-- Searchable input -->
                                <div class="relative">
                                    <input type="text"
                                           id="student_search_input"
                                           placeholder="Type student name or ID..."
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                           autocomplete="off">

                                    <!-- Search results dropdown -->
                                    <div id="student_search_results"
                                         class="absolute z-10 w-full bg-white border border-gray-300 rounded-md shadow-lg max-h-60 overflow-y-auto hidden">
                                        <div id="student_results_list">
                                            <!-- Results will be populated by JavaScript -->
                                        </div>
                                    </div>
                                </div>

                                <!-- Selected student display -->
                                <div id="selected_student_display" class="hidden mt-2 p-3 bg-blue-50 border border-blue-200 rounded-md">
                                    <div class="flex items-center justify-between">
                                        <div>
                                            <h5 class="text-sm font-semibold text-blue-800">Selected Student:</h5>
                                            <div id="selected_student_info" class="text-sm text-blue-700 mt-1"></div>
                                        </div>
                                        <button type="button"
                                                id="clear_student_selection"
                                                class="text-red-600 hover:text-red-800 text-sm">
                                            Clear
                                        </button>
                                    </div>
                                </div>

                                <!-- Hidden fields for form submission -->
                                <input type="hidden" name="assigned_student_id" id="assigned_student_id">
                                <input type="hidden" name="assigned_student_name" id="assigned_student_name">
                            </div>
                        </div>
                        <p class="text-xs text-gray-500 mt-2">Assign this laptop to a specific student for permanent assignment</p>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Registration Date</label>
                        <input type="date" name="purchase_date" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Purchase Price</label>
                        <input type="number" step="0.01" name="purchase_price" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                    </div>
                </div>

                <div class="mt-6 flex justify-end space-x-4">
                    <button type="button" onclick="document.getElementById('addLaptopModal').classList.add('hidden')"
                            class="bg-gray-300 hover:bg-gray-400 text-gray-700 px-4 py-2 rounded-md">
                        Cancel
                    </button>
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-md">
                        Add Laptop
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Student data for searchable dropdown
let allStudents = @json($students);

// Handle searchable student selection for laptop assignment
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('student_search_input');
    const resultsContainer = document.getElementById('student_search_results');
    const resultsList = document.getElementById('student_results_list');
    const selectedDisplay = document.getElementById('selected_student_display');
    const selectedInfo = document.getElementById('selected_student_info');
    const studentIdInput = document.getElementById('assigned_student_id');
    const studentNameInput = document.getElementById('assigned_student_name');
    const clearButton = document.getElementById('clear_student_selection');

    let selectedStudent = null;

    // Search functionality
    searchInput.addEventListener('input', function() {
        const query = this.value.toLowerCase().trim();

        if (query.length < 2) {
            resultsContainer.classList.add('hidden');
            return;
        }

        const filteredStudents = allStudents.filter(student =>
            student.name.toLowerCase().includes(query) ||
            student.id.toLowerCase().includes(query) ||
            student.class_name.toLowerCase().includes(query)
        );

        displaySearchResults(filteredStudents);
        resultsContainer.classList.remove('hidden');
    });

    // Hide results when clicking outside
    document.addEventListener('click', function(e) {
        if (!searchInput.contains(e.target) && !resultsContainer.contains(e.target)) {
            resultsContainer.classList.add('hidden');
        }
    });

    // Clear selection
    clearButton.addEventListener('click', function() {
        clearSelection();
    });

    function displaySearchResults(students) {
        if (students.length === 0) {
            resultsList.innerHTML = '<div class="p-3 text-gray-500 text-sm">No students found</div>';
            return;
        }

        const html = students.slice(0, 10).map(student => `
            <div class="student-result p-3 hover:bg-blue-50 cursor-pointer border-b border-gray-100 last:border-b-0"
                 data-student-id="${student.id}"
                 data-student-name="${student.name}"
                 data-class-name="${student.class_name}">
                <div class="font-medium text-gray-900">${student.name}</div>
                <div class="text-sm text-gray-600">ID: ${student.id} • Class: ${student.class_name}</div>
            </div>
        `).join('');

        resultsList.innerHTML = html;

        // Add click handlers to results
        document.querySelectorAll('.student-result').forEach(result => {
            result.addEventListener('click', function() {
                const studentId = this.getAttribute('data-student-id');
                const studentName = this.getAttribute('data-student-name');
                const className = this.getAttribute('data-class-name');

                selectStudent(studentId, studentName, className);
            });
        });
    }

    function selectStudent(studentId, studentName, className) {
        // Update form fields
        studentIdInput.value = studentId;
        studentNameInput.value = studentName;

        // Update display
        selectedInfo.innerHTML = `
            <div><strong>Name:</strong> ${studentName}</div>
            <div><strong>Student ID:</strong> ${studentId}</div>
            <div><strong>Class:</strong> ${className}</div>
        `;

        // Update search input
        searchInput.value = `${studentName} (${studentId})`;

        // Show selected display, hide results
        selectedDisplay.classList.remove('hidden');
        resultsContainer.classList.add('hidden');

        selectedStudent = { id: studentId, name: studentName, className };
    }

    function clearSelection() {
        studentIdInput.value = '';
        studentNameInput.value = '';
        searchInput.value = '';
        selectedDisplay.classList.add('hidden');
        resultsContainer.classList.add('hidden');
        selectedStudent = null;
    }
});

// Bulk barcode download functionality
document.addEventListener('DOMContentLoaded', function() {
    const selectAllCheckbox = document.getElementById('selectAll');
    const laptopCheckboxes = document.querySelectorAll('.laptop-checkbox');
    const bulkDownloadBtn = document.getElementById('bulkDownloadBtn');

    // Handle select all checkbox
    selectAllCheckbox.addEventListener('change', function() {
        laptopCheckboxes.forEach(checkbox => {
            checkbox.checked = this.checked;
        });
        updateBulkDownloadButton();
    });

    // Handle individual checkbox changes
    laptopCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            updateBulkDownloadButton();

            // Update select all checkbox state
            const checkedBoxes = document.querySelectorAll('.laptop-checkbox:checked');
            selectAllCheckbox.checked = checkedBoxes.length === laptopCheckboxes.length;
            selectAllCheckbox.indeterminate = checkedBoxes.length > 0 && checkedBoxes.length < laptopCheckboxes.length;
        });
    });

    function updateBulkDownloadButton() {
        const checkedBoxes = document.querySelectorAll('.laptop-checkbox:checked');
        if (checkedBoxes.length > 0) {
            bulkDownloadBtn.classList.remove('hidden');
            bulkDownloadBtn.textContent = `Download ${checkedBoxes.length} Barcode${checkedBoxes.length > 1 ? 's' : ''}`;
        } else {
            bulkDownloadBtn.classList.add('hidden');
        }
    }
});

function downloadSelectedBarcodes() {
    const selectedLaptops = document.querySelectorAll('.laptop-checkbox:checked');
    const laptopIds = Array.from(selectedLaptops).map(cb => cb.value);

    if (laptopIds.length === 0) {
        alert('Please select at least one laptop.');
        return;
    }

    // Create form to submit selected laptop IDs
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = '{{ route("laptops.bulk-barcode-pdf") }}';

    // Add CSRF token
    const csrfToken = document.createElement('input');
    csrfToken.type = 'hidden';
    csrfToken.name = '_token';
    csrfToken.value = '{{ csrf_token() }}';
    form.appendChild(csrfToken);

    // Add laptop IDs
    laptopIds.forEach(id => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'laptop_ids[]';
        input.value = id;
        form.appendChild(input);
    });

    document.body.appendChild(form);
    form.submit();
    document.body.removeChild(form);
}
</script>
</script>
@endsection