// resources/js/barcode-scanner.js
class DualBarcodeScanner {
    constructor(config) {
        this.inputField = config.inputField;
        this.resultCallback = config.onResult || this.defaultCallback;
        this.errorCallback = config.onError || this.defaultErrorCallback;
        this.html5QrcodeScanner = null;
        this.scannerBuffer = '';
        this.scannerTimeout = null;
        this.inputTimeout = null;
        this.lastKeyTime = 0;
        this.keyCount = 0;
        
        this.initializeHandheldScanner();
        this.initializeCameraScanner();
    }

    // Static method to check browser support
    static checkBrowserSupport() {
        return !!(navigator.mediaDevices && navigator.mediaDevices.getUserMedia);
    }

    // Handle handheld USB/Bluetooth barcode scanners
    initializeHandheldScanner() {
        // Handheld scanners typically input text very quickly
        this.handleGlobalInput = this.handleGlobalInput.bind(this);
        document.addEventListener('keydown', this.handleGlobalInput);

        // Focus trap for barcode input
        if (this.inputField) {
            this.inputField.addEventListener('input', (event) => {
                this.handleBarcodeInput(event.target.value);
            });

            // Removed auto-focus on page load to allow users to interact with other fields
        }
    }

    handleGlobalInput(event) {
        // Check if we're in a barcode input context
        if (this.isInputFieldFocused() || this.isQuickScan(event)) {
            this.handleHandheldInput(event);
        }
    }

    isInputFieldFocused() {
        return document.activeElement === this.inputField;
    }

    isQuickScan(event) {
        // Detect rapid key sequences typical of barcode scanners
        if (event.key.length === 1) {
            const currentTime = new Date().getTime();
            
            if (currentTime - this.lastKeyTime < 20) {
                this.keyCount++;
            } else {
                this.keyCount = 1;
            }
            
            this.lastKeyTime = currentTime;
            
            clearTimeout(this.scannerTimeout);
            this.scannerBuffer += event.key;
            
            this.scannerTimeout = setTimeout(() => {
                if (this.scannerBuffer.length > 5) { // Minimum barcode length
                    this.processBarcodeFromHandheld(this.scannerBuffer);
                }
                this.scannerBuffer = '';
                this.keyCount = 0;
            }, 100); // Scanner input is usually complete within 100ms
        }
        
        if (event.key === 'Enter' && this.scannerBuffer.length > 0) {
            event.preventDefault();
            this.processBarcodeFromHandheld(this.scannerBuffer);
            this.scannerBuffer = '';
            this.keyCount = 0;
        }
        
        return this.keyCount > 3; // Likely from scanner if more than 3 rapid keys
    }

    handleHandheldInput(event) {
        // Process handheld scanner input
        this.isQuickScan(event);
    }

    processBarcodeFromHandheld(barcode) {
        if (this.inputField) {
            this.inputField.value = barcode;
            this.inputField.dispatchEvent(new Event('input', { bubbles: true }));
            // Add visual feedback for scanner input
            this.inputField.style.borderColor = '#10B981'; // Green border
            setTimeout(() => {
                this.inputField.style.borderColor = '';
            }, 1000);
        }
        this.resultCallback(barcode, 'handheld');
    }

    // Handle camera-based scanning
    initializeCameraScanner() {
        // This integrates with your existing Html5QrcodeScanner
        this.cameraButton = document.getElementById('camera-scan-btn');
        if (this.cameraButton) {
            this.cameraButton.addEventListener('click', () => {
                this.startCameraScanner();
            });
        }
    }

    startCameraScanner() {
        // Check camera availability first
        if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
            this.errorCallback('Camera scanning not supported. Please use your USB barcode scanner or enter the barcode manually.');
            return;
        }

        // Check for HTTPS requirement
        if (location.protocol !== 'https:' && location.hostname !== 'localhost' && location.hostname !== '127.0.0.1') {
            this.errorCallback('Camera scanning requires HTTPS for security. Please use your USB barcode scanner or enter the barcode manually.');
            return;
        }

        const modal = document.getElementById('barcodeModal');
        if (modal) {
            modal.classList.remove('hidden');
            
            // Check if Html5QrcodeScanner is available
            if (typeof Html5QrcodeScanner === 'undefined') {
                this.errorCallback('Camera library not loaded. Please refresh the page and try again.');
                return;
            }
            
            this.html5QrcodeScanner = new Html5QrcodeScanner(
                "qr-reader",
                { 
                    fps: 10, 
                    qrbox: { width: 250, height: 250 },
                    rememberLastUsedCamera: true,
                    supportedScanTypes: [Html5QrcodeScanType.SCAN_TYPE_CAMERA],
                    verbose: false,
                    videoConstraints: {
                        facingMode: "environment" // Use back camera on mobile
                    }
                }
            );
            
            this.html5QrcodeScanner.render(
                (decodedText, decodedResult) => this.onCameraScanSuccess(decodedText, decodedResult),
                (error) => this.onCameraScanFailure(error)
            );
        }
    }

    onCameraScanSuccess(decodedText, decodedResult) {
        if (this.inputField) {
            this.inputField.value = decodedText;
            // Add visual feedback for camera input
            this.inputField.style.borderColor = '#3B82F6'; // Blue border
            setTimeout(() => {
                this.inputField.style.borderColor = '';
            }, 1000);
        }
        this.closeCameraScanner();
        this.resultCallback(decodedText, 'camera');
    }

    onCameraScanFailure(error) {
        // Don't log every scan failure as it's noisy
        // console.warn(`Camera scan failed: ${error}`);
    }

    closeCameraScanner() {
        if (this.html5QrcodeScanner) {
            this.html5QrcodeScanner.clear().then(() => {
                console.log("Scanner cleared successfully");
            }).catch(error => {
                console.error("Error clearing scanner:", error);
            });
        }
        const modal = document.getElementById('barcodeModal');
        if (modal) {
            modal.classList.add('hidden');
        }
    }

    handleBarcodeInput(barcode) {
        if (barcode && barcode.length > 5) {
            // Debounce the input to avoid multiple calls
            clearTimeout(this.inputTimeout);
            this.inputTimeout = setTimeout(() => {
                this.resultCallback(barcode, 'manual');
            }, 300);
        }
    }

    // Auto-focus management for continuous scanning
    keepFocusOnBarcodeInput() {
        if (this.inputField && document.activeElement !== this.inputField) {
            this.inputField.focus();
        }
    }

    startAutoFocus() {
        this.autoFocusInterval = setInterval(() => {
            this.keepFocusOnBarcodeInput();
        }, 2000);
    }

    stopAutoFocus() {
        if (this.autoFocusInterval) {
            clearInterval(this.autoFocusInterval);
        }
    }

    defaultCallback(barcode, source) {
        console.log(`Barcode scanned from ${source}: ${barcode}`);
    }

    defaultErrorCallback(error) {
        console.error('Scanner error:', error);
        alert('Scanner error: ' + error);
    }

    // Utility methods
    static checkBrowserSupport() {
        if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
            console.error('Camera not supported in this browser');
            return false;
        }
        
        if (location.protocol !== 'https:' && location.hostname !== 'localhost') {
            console.error('HTTPS required for camera access');
            return false;
        }
        
        return true;
    }

    static async requestCameraPermission() {
        try {
            const result = await navigator.permissions.query({ name: 'camera' });
            
            if (result.state === 'granted') {
                return true;
            } else if (result.state === 'prompt') {
                // Request permission
                const stream = await navigator.mediaDevices.getUserMedia({ video: true });
                stream.getTracks().forEach(track => track.stop());
                return true;
            } else {
                return false;
            }
        } catch (error) {
            console.error('Permission request failed:', error);
            return false;
        }
    }
}

// Helper functions for status colors and error handling
function getStatusColor(status) {
    switch(status) {
        case 'available': return 'bg-green-100 text-green-800';
        case 'checked_out': return 'bg-yellow-100 text-yellow-800';
        case 'maintenance': return 'bg-red-100 text-red-800';
        default: return 'bg-gray-100 text-gray-800';
    }
}

function showToast(message, type = 'info') {
    // Simple toast notification - you can enhance this
    const toast = document.createElement('div');
    toast.className = `fixed top-4 right-4 p-4 rounded-md shadow-lg z-50 ${
        type === 'success' ? 'bg-green-500 text-white' :
        type === 'error' ? 'bg-red-500 text-white' :
        type === 'warning' ? 'bg-yellow-500 text-white' :
        'bg-blue-500 text-white'
    }`;
    toast.textContent = message;
    
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.remove();
    }, 3000);
}

// Export for global use
if (typeof window !== 'undefined') {
    window.DualBarcodeScanner = DualBarcodeScanner;
    window.getStatusColor = getStatusColor;
    window.showToast = showToast;
}