<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Student;
use App\Models\ClassRoom;

class StudentController extends Controller
{
    public function index(Request $request)
    {
        $query = Student::with(['class', 'currentAssignment.laptop']);

        if ($request->has('class_id')) {
            $query->where('class_id', $request->class_id);
        }

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('student_id', 'like', "%{$search}%");
            });
        }

        $students = $query->paginate(20)->withQueryString();
        $classes = ClassRoom::all();

        return view('students.index', compact('students', 'classes'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'student_id' => 'required|unique:students',
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:students',
            'class_id' => 'required|exists:classes,id'
        ]);

        Student::create($validated);

        return redirect()->route('students.index')
            ->with('success', 'Student added successfully!');
    }

    public function show(Student $student)
    {
        $student->load(['class', 'transactions.laptop', 'currentAssignment.laptop']);
        return view('students.show', compact('student'));
    }

    public function edit(Student $student)
    {
        $student->load('class');
        $classes = ClassRoom::all();
        return view('students.edit', compact('student', 'classes'));
    }

    public function update(Request $request, Student $student)
    {
        $validated = $request->validate([
            'student_id' => 'required|unique:students,student_id,' . $student->id,
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:students,email,' . $student->id,
            'class_id' => 'required|exists:classes,id',
            'is_active' => 'boolean'
        ]);

        $student->update($validated);

        return redirect()->route('students.show', $student)
            ->with('success', 'Student updated successfully!');
    }

    public function destroy(Student $student)
    {
        if ($student->currentAssignment) {
            return back()->withErrors(['student' => 'Cannot delete student with active laptop assignment']);
        }

        $student->delete();

        return redirect()->route('students.index')
            ->with('success', 'Student deleted successfully!');
    }

    public function search(Request $request)
    {
        $studentId = $request->query('student_id');

        if (!$studentId) {
            return response()->json(['error' => 'Student ID is required'], 400);
        }

        $student = Student::where('student_id', $studentId)->with(['class', 'currentAssignment.laptop'])->first();

        if (!$student) {
            return response()->json(['error' => 'Student not found'], 404);
        }

        return response()->json([
            'id' => $student->id,
            'student_id' => $student->student_id,
            'first_name' => $student->first_name,
            'last_name' => $student->last_name,
            'full_name' => $student->full_name,
            'email' => $student->email,
            'is_active' => $student->is_active,
            'has_laptop' => $student->currentAssignment !== null,
            'current_assignment' => $student->currentAssignment ? [
                'laptop' => $student->currentAssignment->laptop,
                'checked_out_at' => $student->currentAssignment->checked_out_at
            ] : null,
            'class' => $student->class
        ]);
    }
}