<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Laptop;
use App\Models\ClassRoom;
use App\Models\Transaction;

// app/Http/Controllers/ReportController.php
class ReportController extends Controller
{
    public function index()
    {
        return view('reports.index');
    }

    public function laptopStatus(Request $request)
    {
        $laptops = Laptop::with(['class', 'currentAssignment.student'])
                         ->orderBy('class_id')
                         ->orderBy('asset_tag')
                         ->paginate(20)
                         ->withQueryString();

        return view('reports.laptop-status', compact('laptops'));
    }

    public function classReport($classId, Request $request)
    {
        $class = ClassRoom::findOrFail($classId);
        $laptops = $class->laptops()->with(['currentAssignment.student'])->paginate(20)->withQueryString();

        return view('reports.class-report', compact('class', 'laptops'));
    }

    public function transactionHistory(Request $request)
    {
        $query = Transaction::with(['laptop', 'student', 'user'])
                           ->orderBy('transaction_time', 'desc');

        if ($request->has('date_from')) {
            $query->whereDate('transaction_time', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->whereDate('transaction_time', '<=', $request->date_to);
        }

        $transactions = $query->paginate(50)->withQueryString();

        return view('reports.transaction-history', compact('transactions'));
    }

    public function exportPdf($type, Request $request)
    {
        $data = [];
        $view = '';
        $filename = '';

        switch ($type) {
            case 'laptop-status':
                $data['laptops'] = Laptop::with(['class', 'currentAssignment.student'])
                                        ->orderBy('class_id')
                                        ->orderBy('asset_tag')
                                        ->get();
                $view = 'reports.pdf.laptop-status';
                $filename = 'laptop-status-report-' . date('Y-m-d') . '.pdf';
                break;

            case 'transaction-history':
                $query = Transaction::with(['laptop', 'student', 'user'])
                                   ->orderBy('transaction_time', 'desc');

                if ($request->has('date_from')) {
                    $query->whereDate('transaction_time', '>=', $request->date_from);
                }

                if ($request->has('date_to')) {
                    $query->whereDate('transaction_time', '<=', $request->date_to);
                }

                $data['transactions'] = $query->get();
                $data['date_from'] = $request->date_from;
                $data['date_to'] = $request->date_to;
                $view = 'reports.pdf.transaction-history';
                $filename = 'transaction-history-report-' . date('Y-m-d') . '.pdf';
                break;

            case 'class':
                if (!$request->has('class_id')) {
                    return back()->withErrors(['class' => 'Class ID is required for class report']);
                }

                $class = ClassRoom::findOrFail($request->class_id);
                $data['class'] = $class;
                $data['laptops'] = $class->laptops()->with(['currentAssignment.student'])->get();
                $view = 'reports.pdf.class-report';
                $filename = 'class-report-' . $class->name . '-' . date('Y-m-d') . '.pdf';
                break;

            default:
                return back()->withErrors(['type' => 'Invalid report type']);
        }

        if (empty($view)) {
            return back()->withErrors(['type' => 'Report type not implemented']);
        }

        $pdf = app('dompdf.wrapper');
        $pdf->loadView($view, $data);

        return $pdf->download($filename);
    }
}