<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Models\Laptop;
use App\Models\Transaction;

class DashboardController extends Controller
{
    /**
     * Display the dashboard with laptop statistics
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        // Calculate laptop statistics
        $stats = [
            'total_laptops' => Laptop::count(),
            'available_laptops' => Laptop::where('status', 'available')->count(),
            'checked_out_laptops' => Laptop::where('status', 'checked_out')->count(),
            'maintenance_laptops' => Laptop::where('status', 'maintenance')->count(),
        ];

        return view('dashboard', compact('stats'));
    }

    /**
     * Process barcode scan and return laptop information
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function processBarcodeScann(Request $request): JsonResponse
    {
        $request->validate([
            'barcode' => 'required|string'
        ]);

        $barcode = $request->input('barcode');
        
        // Find laptop by barcode
        $laptop = Laptop::where('barcode', $barcode)->first();
        
        if (!$laptop) {
            return response()->json([
                'error' => 'Laptop not found with this barcode'
            ], 404);
        }

        $response = [
            'laptop_id' => $laptop->id,
            'laptop_barcode' => $laptop->barcode,
            'laptop_model' => $laptop->model ?? null,
            'laptop_serial' => $laptop->serial_number ?? null,
            'is_checked_out' => $laptop->status === 'checked_out',
            'status' => $laptop->status
        ];

        // If laptop is checked out, get current student information
        if ($laptop->status === 'checked_out') {
            $currentTransaction = Transaction::where('laptop_id', $laptop->id)
                ->where('status', 'active')
                ->with('student')
                ->first();
                
            if ($currentTransaction && $currentTransaction->student) {
                $response['current_student'] = [
                    'id' => $currentTransaction->student->id,
                    'full_name' => $currentTransaction->student->first_name . ' ' . $currentTransaction->student->last_name,
                    'student_id' => $currentTransaction->student->student_id
                ];
            }
        }

        return response()->json($response);
    }
}