# Laptop Tracking System

## Software Design Document

**Author:** Martin Mugenyi

**Project Title:** Laptop Tracking System

**Version:** 1.0

**Workstation:** Local Development Environment

**Date:** 09/05/2025

---

## TABLE OF CONTENTS

1. [INTRODUCTION](#1-introduction)
   1. [Purpose](#11-purpose)
   2. [Scope](#12-scope)
   3. [Overview](#13-overview)
   4. [Reference Material](#14-reference-material)
   5. [Definitions and Acronyms](#15-definitions-and-acronyms)
2. [SYSTEM OVERVIEW](#2-system-overview)
3. [SYSTEM ARCHITECTURE](#3-system-architecture)
   1. [Architectural Design](#31-architectural-design)
   2. [Decomposition Description](#32-decomposition-description)
   3. [Design Rationale](#33-design-rationale)
   4. [Design Patterns Used](#34-design-patterns-used)
4. [DATA DESIGN](#4-data-design)
   1. [Data Description](#41-data-description)
   2. [Data Dictionary](#42-data-dictionary)
5. [COMPONENT DESIGN](#5-component-design)
   1. [Component Architecture Overview](#51-component-architecture-overview)
   2. [Controller Components](#52-controller-components)
   3. [Service Components](#53-service-components)
   4. [Model Components](#54-model-components)
6. [HUMAN INTERFACE DESIGN](#6-human-interface-design)
   1. [Overview of User Interface](#61-overview-of-user-interface)
   2. [Screen Images](#62-screen-images)
   3. [Screen Objects and Actions](#63-screen-objects-and-actions)
7. [REQUIREMENTS MATRIX](#7-requirements-matrix)
8. [APPENDICES](#8-appendices)
   1. [Database Schema](#appendix-a-database-schema)
   2. [API Endpoints](#appendix-b-api-endpoints)
   3. [Technology Stack](#appendix-c-technology-stack)
   4. [System Constraints and Assumptions](#appendix-d-system-constraints-and-assumptions)
   5. [Performance Requirements](#appendix-e-performance-requirements)
   6. [Security Considerations](#appendix-f-security-considerations)

---

# 1. INTRODUCTION

## 1.1 Purpose

This Software Design Document (SDD) describes the architecture and system design of the Laptop Tracking System. This document serves as the primary reference for code development and contains all the information required by a programmer to write code. It is intended for developers, system architects, and stakeholders involved in the implementation and maintenance of the system.

## 1.2 Scope

The Laptop Tracking System is a web-based application built using the Laravel framework that manages the inventory, assignment, and tracking of laptops in an educational institution. The system provides functionality to:

- Manage laptop inventory with detailed specifications
- Track student information and class assignments
- Assign laptops to students and record transactions
- Generate barcodes and QR codes for easy identification
- Generate reports on laptop status, assignments, and usage history
- Provide a user-friendly web interface for staff operations

The system aims to streamline laptop management, reduce loss, and improve accountability in laptop distribution within the institution.

## 1.3 Overview

This document is organized as follows: Section 2 provides a general overview of the system. Section 3 describes the system architecture and decomposition. Section 4 details the data design and structures. Section 5 provides component-level design details. Section 6 describes the human interface design. Section 7 provides a requirements traceability matrix. Section 8 contains appendices.

## 1.4 Reference Material

- Laravel Framework Documentation: https://laravel.com/docs
- IEEE Recommended Practice for Software Design Descriptions (IEEE Std 1016-1998)
- Project codebase and database schema

## 1.5 Definitions and Acronyms

- **MVC**: Model-View-Controller architectural pattern
- **CRUD**: Create, Read, Update, Delete operations
- **ORM**: Object-Relational Mapping
- **API**: Application Programming Interface
- **PDF**: Portable Document Format
- **QR Code**: Quick Response Code

# 2. SYSTEM OVERVIEW

The Laptop Tracking System is a web-based application designed to manage the lifecycle of laptops in an educational environment. The system tracks laptop inventory, student assignments, and transaction history to ensure accountability and efficient resource management.

The system consists of several key components:
- User authentication and authorization
- Laptop inventory management
- Student and class management
- Transaction tracking (check-in/check-out)
- Barcode and QR code generation
- Reporting functionality
- PDF generation for labels and reports

The application is built using Laravel, a PHP framework, with a MySQL database backend. It uses modern web technologies for the frontend and provides both web interface and API endpoints for integration.

# 3. SYSTEM ARCHITECTURE

## 3.1 Architectural Design

The system follows a Model-View-Controller (MVC) architectural pattern implemented through the Laravel framework. The high-level architecture consists of the following subsystems:

1. **Presentation Layer**: Handles user interface and HTTP requests
2. **Application Layer**: Contains business logic and controllers
3. **Data Layer**: Manages data persistence and ORM
4. **Services Layer**: Provides utility services like barcode generation

### System Architecture Diagram

```
+---------------------------------------------------------------------+
|                    LAPTOP TRACKING SYSTEM                           |
+---------------------------------------------------------------------+
|                                                                     |
|  +----------------+  +----------------+  +----------------------+   |
|  |                |  |                |  |                      |   |
|  | Authentication |  | Laptop         |  | Transaction          |   |
|  | Management     |  | Management     |  | Management           |   |
|  | Subsystem      |  | Subsystem      |  | Subsystem            |   |
|  |                |  |                |  |                      |   |
|  +----------------+  +----------------+  +----------------------+   |
|                                                                     |
|  +----------------+  +----------------+  +----------------------+   |
|  |                |  |                |  |                      |   |
|  | Student        |  | Class          |  | Reporting            |   |
|  | Management     |  | Management     |  | Subsystem            |   |
|  | Subsystem      |  | Subsystem      |  |                      |   |
|  |                |  |                |  |                      |   |
|  +----------------+  +----------------+  +----------------------+   |
|                                                                     |
|  +----------------+  +----------------+  +----------------------+   |
|  |                |  |                |  |                      |   |
|  | Barcode        |  | PDF Generation |  | Notification         |   |
|  | Services       |  | Services       |  | Services             |   |
|  | Subsystem      |  | Subsystem      |  | Subsystem            |   |
|  |                |  |                |  |                      |   |
|  +----------------+  +----------------+  +----------------------+   |
|                                                                     |
+---------------------------------------------------------------------+
```

**Figure 1: System Architecture Overview**

### Subsystem Descriptions

1. **Authentication Management Subsystem**
   - Handles user registration, login, and session management
   - Implements role-based access control
   - Manages user profiles and permissions

2. **Laptop Management Subsystem**
   - Manages laptop inventory with detailed specifications
   - Handles CRUD operations for laptops
   - Manages laptop status and assignments

3. **Transaction Management Subsystem**
   - Records check-in/check-out transactions
   - Tracks laptop assignments to students
   - Maintains transaction history and audit trails

4. **Student Management Subsystem**
   - Manages student information and profiles
   - Handles student-class associations
   - Tracks student activity and assignments

5. **Class Management Subsystem**
   - Manages class information and structure
   - Handles grade levels and sections
   - Maintains class-student relationships

6. **Reporting Subsystem**
   - Generates individual, departmental, and institutional reports
   - Provides performance analytics and visualizations
   - Exports reports in various formats (PDF, Excel)

7. **Barcode Services Subsystem**
   - Generates barcodes and QR codes for laptops
   - Manages barcode uniqueness and validation
   - Integrates with external barcode libraries

8. **PDF Generation Services Subsystem**
   - Creates PDF documents for barcodes and labels
   - Generates bulk barcode PDFs
   - Handles PDF formatting and layout

9. **Notification Services Subsystem**
   - Sends email notifications for key events
   - Provides in-app notifications and reminders
   - Alerts users about pending tasks and deadlines

### Component Architecture Diagram

```mermaid
graph TD
    A[User] --> B[Web Browser]
    B --> C[Laravel Application]
    C --> D[HTTP Routes]
    D --> E[Controllers]
    E --> F[Models]
    E --> G[Services]
    F --> H[Database]
    G --> I[Barcode/QR Generation]
    G --> J[PDF Generation]
    E --> K[Views]
    K --> B
    H --> L[(MySQL Database)]
    E --> M[Middleware]
    M --> N[Authentication]
```

**Figure 2: Component Architecture Diagram**

## 3.2 Decomposition Description

### Functional Decomposition

The system uses an object-oriented approach with the following key components:

1. **Controllers**:
   - `LaptopController`: Handles laptop CRUD and barcode generation
   - `StudentController`: Manages student information
   - `ClassRoomController`: Manages class information
   - `TransactionController`: Handles check-in/check-out operations
   - `ReportController`: Generates various reports

2. **Models**:
   - `Laptop`: Represents laptop entities
   - `Student`: Represents student entities
   - `ClassRoom`: Represents class entities
   - `Transaction`: Represents transaction records
   - `CurrentAssignment`: Tracks current laptop assignments

3. **Services**:
   - `BarcodeService`: Generates barcodes and QR codes

### Data Flow Diagram

```mermaid
graph TD
    A[User Interface] --> B[Controller]
    B --> C[Model]
    C --> D[Database]
    B --> E[Service Layer]
    E --> F[External Libraries]
    D --> C
    C --> B
    B --> A

    subgraph "Business Logic Layer"
    B
    E
    end

    subgraph "Data Access Layer"
    C
    D
    end

    subgraph "Presentation Layer"
    A
    end
```

**Figure 5: Detailed Data Flow Diagram**

### Sequence Diagram - Laptop Checkout Process

```
User -> Controller: Request Checkout
Controller -> Model: Validate Laptop
Model -> Database: Check Availability
Database --> Model: Return Status
Model --> Controller: Laptop Available
Controller -> Service: Generate Transaction
Service -> Database: Create Transaction
Database --> Service: Transaction Created
Service --> Controller: Success Response
Controller -> View: Render Success Page
View --> User: Display Confirmation
```

**Figure 6: Laptop Checkout Sequence Diagram**

## 3.3 Design Rationale

The MVC architecture was chosen for its separation of concerns, making the codebase maintainable and scalable. Laravel was selected for its robust features, security, and rapid development capabilities. The relational database design ensures data integrity and efficient querying.

Alternative architectures considered:
- **Microservices**: Rejected due to project scope and complexity
- **SPA Framework**: Not necessary for current requirements
- **NoSQL Database**: Relational database better suited for structured data

The design prioritizes:
- **Maintainability**: Clear separation of concerns
- **Scalability**: Modular design allows for future expansion
- **Security**: Built-in Laravel security features
- **Usability**: Intuitive web interface

## 3.4 Design Patterns Used

The system implements several design patterns to enhance maintainability, scalability, and code organization:

### 1. MVC Pattern (Model-View-Controller)
- **Model**: Represents data and business logic (Eloquent models)
- **View**: Handles presentation layer (Blade templates)
- **Controller**: Manages user input and application flow

### 2. Repository Pattern
- Abstract data access layer through Eloquent ORM
- Centralized data access logic
- Easier testing and maintenance

### 3. Service Layer Pattern
- `BarcodeService` encapsulates barcode generation logic
- Separation of business logic from controllers
- Improved testability and reusability

### 4. Factory Pattern
- Auto-generation of asset tags and barcodes
- Centralized object creation logic
- Consistent data generation rules

### 5. Observer Pattern
- Model events for audit trails and notifications
- Automatic actions on model state changes
- Loose coupling between components

### 6. Strategy Pattern
- Multiple barcode generation strategies (Code 128, QR Code)
- Flexible report generation formats
- Easy extension for new algorithms

### 7. Singleton Pattern
- Database connection management
- Service container management
- Configuration management

### 8. Dependency Injection
- Laravel service container for dependency management
- Constructor injection in controllers and services
- Improved testability and flexibility

# 4. DATA DESIGN

## 4.1 Data Description

The system uses a relational database to store all persistent data. The main entities are:

- **Laptops**: Store detailed information about each laptop including specifications, status, and assignments
- **Students**: Store student information and class associations
- **Classes**: Store class information and relationships
- **Transactions**: Store historical records of laptop assignments and returns
- **Users**: Store system user information for authentication

Data is organized using foreign key relationships to maintain referential integrity. The system uses Laravel's Eloquent ORM for database interactions.

### Entity Relationship Diagram (ERD)

```
+------------+       +------------+       +------------+
|   Users    |       |  Classes   |       |  Students  |
+------------+       +------------+       +------------+
| id         |<--+   | id         |<--+   | id         |
| name       |   |   | name       |   |   | student_id |
| email      |   |   | grade_level|   +---| first_name |
| password   |   |   | section    |       | last_name  |
| ...        |   |   | description|       | email      |
+------------+   |   | is_active  |       | class_id   |
                 |   +------------+       | is_active  |
                 |                        +------------+
                 |
                 |   +------------+       +------------+
                 |   |  Laptops   |       |Transactions|
                 |   +------------+       +------------+
                 +---| id         |<--+   | id         |
                     | asset_tag  |   |   | laptop_id  |
                     | barcode    |   |   | student_id |
                     | brand      |   |   | user_id    |
                     | model      |   |   | type       |
                     | serial_no  |   |   | trans_time |
                     | status     |   |   | notes      |
                     | class_id   |   |   | condition_b|
                     | student_id |   |   | condition_a|
                     | notes      |   |   +------------+
                     | purch_date |
                     | purch_price|
                     +------------+

                     +------------+
                     |CurrentAssign|
                     +------------+
                     | id         |
                     | laptop_id  |
                     | student_id |
                     | checked_out|
                     | checked_by |
                     +------------+
```

**Figure 3: Entity Relationship Diagram**

### Database Relationships

- **Users** 1:N **Transactions** (One user can perform many transactions)
- **Classes** 1:N **Students** (One class can have many students)
- **Classes** 1:N **Laptops** (One class can have many laptops assigned)
- **Students** 1:N **Transactions** (One student can have many transactions)
- **Laptops** 1:N **Transactions** (One laptop can have many transaction records)
- **Laptops** 1:1 **CurrentAssignment** (One laptop can have one current assignment)
- **Students** 1:1 **CurrentAssignment** (One student can have one current assignment)

## 4.2 Data Dictionary

### Laptop
- **id**: Primary key (integer, auto-increment)
- **asset_tag**: Unique asset identifier (string)
- **barcode**: Unique barcode (string)
- **brand**: Laptop brand (string)
- **model**: Laptop model (string)
- **serial_number**: Manufacturer serial number (string, unique)
- **status**: Current status (enum: available, checked_out, maintenance, damaged, retired)
- **class_id**: Foreign key to classes table (integer)
- **assigned_student_id**: Student ID if assigned (string, nullable)
- **assigned_student_name**: Student name if assigned (string, nullable)
- **notes**: Additional notes (text, nullable)
- **purchase_date**: Date of purchase (date, nullable)
- **purchase_price**: Purchase price (decimal, nullable)
- **created_at**: Creation timestamp
- **updated_at**: Update timestamp

### Student
- **id**: Primary key (integer, auto-increment)
- **student_id**: Unique student identifier (string)
- **first_name**: Student first name (string)
- **last_name**: Student last name (string)
- **email**: Student email (string, nullable)
- **class_id**: Foreign key to classes table (integer)
- **is_active**: Active status (boolean)
- **created_at**: Creation timestamp
- **updated_at**: Update timestamp

### ClassRoom
- **id**: Primary key (integer, auto-increment)
- **name**: Class name (string)
- **grade_level**: Grade level (string)
- **section**: Class section (string)
- **description**: Class description (text, nullable)
- **is_active**: Active status (boolean)
- **created_at**: Creation timestamp
- **updated_at**: Update timestamp

### Transaction
- **id**: Primary key (integer, auto-increment)
- **laptop_id**: Foreign key to laptops table (integer)
- **student_id**: Foreign key to students table (integer)
- **user_id**: Foreign key to users table (integer)
- **type**: Transaction type (enum: check_out, check_in)
- **transaction_time**: Timestamp of transaction (datetime)
- **notes**: Transaction notes (text, nullable)
- **condition_before**: Laptop condition before transaction (string, nullable)
- **condition_after**: Laptop condition after transaction (string, nullable)
- **created_at**: Creation timestamp
- **updated_at**: Update timestamp

### CurrentAssignment
- **id**: Primary key (integer, auto-increment)
- **laptop_id**: Foreign key to laptops table (integer)
- **student_id**: Foreign key to students table (integer)
- **checked_out_at**: Checkout timestamp (datetime)
- **checked_out_by**: User who performed checkout (integer)
- **created_at**: Creation timestamp
- **updated_at**: Update timestamp

# 5. COMPONENT DESIGN

## 5.1 Component Architecture Overview

The system is built using a layered architecture with clear separation of concerns. Components are organized into logical layers that promote maintainability and scalability.

### Component Interaction Diagram

```
+----------------+     +----------------+     +----------------+
|   Controllers  | --> |    Services    | --> |   External     |
|                |     |                |     |   Libraries    |
| - LaptopCtrl   |     | - BarcodeSvc   |     | - DomPDF       |
| - StudentCtrl  |     | - ReportSvc    |     | - Endroid QR   |
| - ClassCtrl    |     | - AuthSvc      |     | - Milon Barcode|
| - TransCtrl    |     |                |     |                |
+----------------+     +----------------+     +----------------+
        |                       |
        v                       v
+----------------+     +----------------+
|     Models     |     |   Database     |
|                |     |                |
| - Laptop       |     | - MySQL        |
| - Student      |     | - Migrations   |
| - ClassRoom    |     | - Seeders      |
| - Transaction  |     |                |
+----------------+     +----------------+
        ^
        |
+----------------+
|     Views      |
|                |
| - Blade        |
| - Components   |
| - Layouts      |
+----------------+
```

**Figure 4: Component Interaction Diagram**

## 5.2 Controller Components

### LaptopController

**Purpose**: Handles all laptop-related operations including CRUD, barcode generation, and API endpoints.

**Key Methods**:
- `index()`: Display paginated list of laptops with filtering
- `store()`: Create new laptop with validation
- `show()`: Display detailed laptop information
- `generateBarcodePdf()`: Generate PDF with barcode for single laptop
- `bulkGenerateBarcodePdf()`: Generate PDF with barcodes for multiple laptops
- `getByBarcode()`: API endpoint to retrieve laptop by barcode

**Algorithm for store()**:
```
1. Validate input data
2. Create laptop record
3. Auto-generate asset tag if not provided
4. Auto-generate barcode if not provided
5. Return success response with asset tag
```

### TransactionController

**Purpose**: Manages laptop check-in/check-out transactions and assignment tracking.

**Key Methods**:
- `checkOut()`: Process laptop checkout to student
- `checkIn()`: Process laptop return from student
- `history()`: Display transaction history
- `validateTransaction()`: Validate transaction rules

## 5.3 Service Components

### BarcodeService

**Purpose**: Provides barcode and QR code generation functionality.

**Key Methods**:
- `generateBarcode()`: Generate barcode image using DNS1D library
- `generateQRCode()`: Generate QR code using Endroid library
- `generateQRCodeForLaptop()`: Create QR code with laptop information

**Algorithm for generateQRCodeForLaptop()**:
```
1. Collect laptop data (asset_tag, barcode, brand, model, etc.)
2. Create JSON structure with laptop information
3. Generate QR code from JSON data
4. Return base64 encoded image
```

### ReportService

**Purpose**: Handles report generation and data analysis.

**Key Methods**:
- `generateLaptopStatusReport()`: Generate laptop status summary
- `generateTransactionHistory()`: Generate transaction history report
- `generateClassReport()`: Generate class-wise laptop assignments

## 5.4 Model Components

### Laptop Model

**Purpose**: Represents laptop entity with business logic.

**Key Methods**:
- `isAvailable()`: Check if laptop is available for assignment
- `generateAssetTag()`: Auto-generate unique asset tag
- `generateBarcode()`: Auto-generate unique barcode

**Relationships**:
- Belongs to ClassRoom
- Has many Transactions
- Has one CurrentAssignment

### Transaction Model

**Purpose**: Represents transaction records with audit trail.

**Key Methods**:
- `getDuration()`: Calculate transaction duration
- `validateCondition()`: Validate laptop condition changes

**Relationships**:
- Belongs to Laptop
- Belongs to Student
- Belongs to User

# 6. HUMAN INTERFACE DESIGN

## 6.1 Overview of User Interface

The system provides a web-based interface built with Laravel Blade templates and Tailwind CSS. The interface includes:

- Dashboard with system overview
- Laptop management pages (list, create, edit, view)
- Student management pages
- Class management pages
- Transaction history
- Report generation
- Barcode generation tools

The interface uses Jetstream for authentication and Livewire for dynamic components. Users can perform all operations through the web interface, with responsive design for mobile access.

## 6.2 Screen Images

### Dashboard Screen
```
+-----------------------------------+
|         LAPTOP TRACKING          |
|         SYSTEM DASHBOARD         |
+-----------------------------------+
| Total Laptops: 150               |
| Available: 120                   |
| Checked Out: 25                  |
| Maintenance: 5                   |
+-----------------------------------+
| Recent Transactions              |
| - Laptop LT-2025-001 checked out |
| - Laptop LT-2025-002 returned    |
| ...                              |
+-----------------------------------+
```

### Laptop List Screen
```
+-----------------------------------+
|         LAPTOP INVENTORY         |
+-----------------------------------+
| [Filter by Class] [Filter by Status]
|                                   |
| Asset Tag | Brand | Model | Status |
| LT-2025-001 | Dell | XPS 13 | Available |
| LT-2025-002 | HP | EliteBook | Checked Out |
| ...                              |
| [Add New Laptop] [Generate Barcodes]
+-----------------------------------+
```

### Laptop Detail Screen
```
+-----------------------------------+
|     LAPTOP DETAILS               |
+-----------------------------------+
| Asset Tag: LT-2025-001           |
| Barcode: [BARCODE IMAGE]         |
| Brand: Dell                      |
| Model: XPS 13                    |
| Serial: ABC123                   |
| Status: Available                |
| Class: Grade 10A                 |
|                                   |
| Transaction History              |
| Date | Type | Student | User     |
| 2025-01-01 | Check Out | John Doe | Admin |
| ...                              |
+-----------------------------------+
```

## 6.3 Screen Objects and Actions

### Navigation Menu
- **Dashboard**: Link to main dashboard
- **Laptops**: Dropdown with List, Add New
- **Students**: Dropdown with List, Add New
- **Classes**: Dropdown with List, Add New
- **Reports**: Dropdown with various report types
- **Profile**: User profile management

### Form Elements
- **Text Input**: For asset tags, names, etc.
- **Select Dropdown**: For status, class selection
- **Date Picker**: For purchase dates
- **File Upload**: For bulk operations
- **Submit Button**: To save changes
- **Cancel Button**: To discard changes

### Action Buttons
- **Edit**: Modify existing records
- **Delete**: Remove records (with confirmation)
- **View Details**: Show detailed information
- **Generate Barcode**: Create barcode PDF
- **Check Out/In**: Process transactions

# 7. REQUIREMENTS MATRIX

| Requirement ID | Description | Component | Data Structure | Status |
|----------------|-------------|-----------|----------------|--------|
| FR-1 | Manage laptop inventory | LaptopController, Laptop Model | laptops table | Implemented |
| FR-2 | Track student information | StudentController, Student Model | students table | Implemented |
| FR-3 | Manage class information | ClassRoomController, ClassRoom Model | classes table | Implemented |
| FR-4 | Record transactions | TransactionController, Transaction Model | transactions table | Implemented |
| FR-5 | Generate barcodes | BarcodeService, LaptopController | laptops.barcode | Implemented |
| FR-6 | Generate reports | ReportController | Various queries | Implemented |
| FR-7 | User authentication | Jetstream | users table | Implemented |
| FR-8 | PDF generation | DomPDF integration | N/A | Implemented |
| FR-9 | API endpoints | Controllers | JSON responses | Implemented |
| FR-10 | Responsive UI | Blade templates, Tailwind CSS | N/A | Implemented |

# 8. APPENDICES

## Appendix A: Database Schema

### Complete Database Schema

#### Users Table
```sql
CREATE TABLE users (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    email VARCHAR(255) UNIQUE NOT NULL,
    email_verified_at TIMESTAMP NULL,
    password VARCHAR(255) NOT NULL,
    two_factor_secret TEXT NULL,
    two_factor_recovery_codes TEXT NULL,
    remember_token VARCHAR(100) NULL,
    current_team_id BIGINT UNSIGNED NULL,
    profile_photo_path VARCHAR(2048) NULL,
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL
);
```

#### Classes Table
```sql
CREATE TABLE classes (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    grade_level VARCHAR(255) NOT NULL,
    section VARCHAR(255) NOT NULL,
    description TEXT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL
);
```

#### Students Table
```sql
CREATE TABLE students (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    student_id VARCHAR(255) UNIQUE NOT NULL,
    first_name VARCHAR(255) NOT NULL,
    last_name VARCHAR(255) NOT NULL,
    email VARCHAR(255) UNIQUE NULL,
    class_id BIGINT UNSIGNED NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    FOREIGN KEY (class_id) REFERENCES classes(id)
);
```

#### Laptops Table
```sql
CREATE TABLE laptops (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    asset_tag VARCHAR(255) UNIQUE NOT NULL,
    barcode VARCHAR(255) UNIQUE NOT NULL,
    brand VARCHAR(255) NOT NULL,
    model VARCHAR(255) NOT NULL,
    serial_number VARCHAR(255) UNIQUE NOT NULL,
    status ENUM('available', 'checked_out', 'maintenance', 'damaged', 'retired') DEFAULT 'available',
    class_id BIGINT UNSIGNED,
    assigned_student_id VARCHAR(255) NULL,
    assigned_student_name VARCHAR(255) NULL,
    notes TEXT NULL,
    purchase_date DATE NULL,
    purchase_price DECIMAL(10,2) NULL,
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    FOREIGN KEY (class_id) REFERENCES classes(id)
);
```

#### Transactions Table
```sql
CREATE TABLE transactions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    laptop_id BIGINT UNSIGNED NOT NULL,
    student_id BIGINT UNSIGNED NOT NULL,
    user_id BIGINT UNSIGNED NOT NULL,
    type ENUM('check_out', 'check_in') NOT NULL,
    transaction_time TIMESTAMP NOT NULL,
    notes TEXT NULL,
    condition_before VARCHAR(255) NULL,
    condition_after VARCHAR(255) NULL,
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    FOREIGN KEY (laptop_id) REFERENCES laptops(id),
    FOREIGN KEY (student_id) REFERENCES students(id),
    FOREIGN KEY (user_id) REFERENCES users(id)
);
```

#### Current Assignments Table
```sql
CREATE TABLE current_assignments (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    laptop_id BIGINT UNSIGNED UNIQUE NOT NULL,
    student_id BIGINT UNSIGNED NOT NULL,
    checked_out_at TIMESTAMP NOT NULL,
    checked_out_by BIGINT UNSIGNED NOT NULL,
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    FOREIGN KEY (laptop_id) REFERENCES laptops(id),
    FOREIGN KEY (student_id) REFERENCES students(id),
    FOREIGN KEY (checked_out_by) REFERENCES users(id)
);
```

## Appendix B: API Endpoints

### RESTful API Endpoints

#### Laptop Management
- `GET /api/laptops` - List all laptops with pagination and filtering
- `POST /api/laptops` - Create new laptop
- `GET /api/laptops/{id}` - Get laptop details
- `PUT /api/laptops/{id}` - Update laptop information
- `DELETE /api/laptops/{id}` - Delete laptop
- `GET /api/laptops/{barcode}` - Get laptop by barcode (for mobile scanning)

#### Student Management
- `GET /api/students` - List all students
- `POST /api/students` - Create new student
- `GET /api/students/{id}` - Get student details
- `PUT /api/students/{id}` - Update student information

#### Transaction Management
- `POST /api/transactions/check-out` - Check out laptop to student
- `POST /api/transactions/check-in` - Check in laptop from student
- `GET /api/transactions` - List transactions with filtering

#### Reporting
- `GET /api/reports/laptop-status` - Generate laptop status report
- `GET /api/reports/transaction-history` - Generate transaction history
- `GET /api/reports/class-summary` - Generate class-wise summary

## Appendix C: Technology Stack

### Backend Technologies
- **Framework**: Laravel 12.x
- **Language**: PHP 8.2+
- **Database**: MySQL 8.0+
- **ORM**: Eloquent ORM
- **Authentication**: Laravel Jetstream with Fortify

### Frontend Technologies
- **Templates**: Blade templating engine
- **Styling**: Tailwind CSS
- **JavaScript**: Livewire for reactive components
- **Icons**: Heroicons

### External Libraries and Packages
- **PDF Generation**: DomPDF (^3.1)
- **QR Code Generation**: Endroid QR Code (^6.0)
- **Barcode Generation**: Milon Barcode (^12.0)
- **Image Processing**: Intervention Image (^3.11)
- **UI Components**: Laravel Jetstream

### Development Tools
- **Version Control**: Git
- **Package Management**: Composer, NPM
- **Build Tool**: Vite
- **Testing**: PHPUnit, Pest
- **Code Quality**: Laravel Pint

### Server Requirements
- **Web Server**: Apache 2.4+ or Nginx 1.20+
- **PHP Extensions**: PDO, MBString, OpenSSL, GD, DOM, XML
- **Database**: MySQL 8.0+ or MariaDB 10.3+
- **Memory**: Minimum 512MB RAM
- **Storage**: 100MB+ for application files

## Appendix D: System Constraints and Assumptions

### Performance Constraints
- Response time for web pages: < 2 seconds
- API response time: < 500ms
- Concurrent users: Up to 100 simultaneous users
- Database query timeout: 30 seconds
- File upload size limit: 10MB

### Security Constraints
- All user input must be validated and sanitized
- Passwords must be hashed using bcrypt
- Session timeout: 2 hours of inactivity
- Failed login attempts: Maximum 5 before lockout
- Data encryption: Sensitive data encrypted at rest

### Business Rules
- Each laptop can only be assigned to one student at a time
- Students can only have one laptop assigned at a time
- Asset tags and barcodes must be unique
- Transactions must maintain audit trail
- Only authorized staff can perform check-out/check-in operations

### Assumptions
- Internet connectivity is available for web access
- Users have basic computer literacy
- Barcode scanning devices are available for mobile operations
- Database backup is performed regularly by system administrator
- Regular maintenance windows are available for system updates

## Appendix E: Performance Requirements

### Response Time Requirements
- Page load time: < 2 seconds for 95% of requests
- API response time: < 500ms for 95% of requests
- Report generation: < 30 seconds for standard reports
- Barcode generation: < 5 seconds for bulk operations

### Throughput Requirements
- 100 concurrent users during peak hours
- 1000 transactions per hour
- 100 reports generated per day
- 500 barcode scans per hour

### Availability Requirements
- System uptime: 99.5% (excluding planned maintenance)
- Database availability: 99.9%
- Backup completion: Within 4 hours
- Disaster recovery: RTO < 4 hours, RPO < 1 hour

## Appendix F: Security Considerations

### Authentication and Authorization
- Multi-factor authentication for admin users
- Role-based access control (RBAC)
- Session management with secure cookies
- Password complexity requirements

### Data Protection
- Encryption of sensitive data at rest
- SSL/TLS for data in transit
- Input validation and sanitization
- SQL injection prevention through parameterized queries

### Audit and Compliance
- Complete audit trail for all transactions
- User activity logging
- Data retention policies
- Regular security assessments