# Admin User Management System

## Overview
The Laptop Tracking System now includes a comprehensive admin user management system. Admins can create, edit, and delete user accounts, and track login activities across the system. Public registration has been disabled - only administrators can create new user accounts.

## Features Implemented

### 1. Role-Based Access Control
- **Admin Role**: Full access to user management features
- **User Role**: Standard access to laptop tracking features
- Users cannot self-register; only admins can create accounts

### 2. User Management
Admins can:
- View all users with their details
- Create new user accounts with custom roles
- Edit existing user information
- Activate/deactivate user accounts
- Delete users (except themselves)
- View login activity for each user

### 3. Login Activity Tracking
The system automatically logs:
- Login timestamps
- IP addresses
- Browser/device information (user agent)
- Login count per user

### 4. Security Features
- Public registration disabled
- Admin middleware protection on user routes
- Inactive users are automatically logged out
- Admins cannot delete their own accounts
- Password confirmation required for new/changed passwords

## Database Schema Changes

### Users Table Additions
```sql
- role (enum: 'admin', 'user') - Default: 'user'
- is_active (boolean) - Default: true
- last_login_at (timestamp) - Nullable
```

### New Table: user_activity_logs
```sql
- id (bigint, primary key)
- user_id (foreign key to users)
- action (varchar) - Default: 'login'
- ip_address (varchar, 45)
- user_agent (text)
- created_at (timestamp)
```

## Default Admin Account

After running migrations and seeders, use these credentials:

**Admin Account:**
- Email: `admin@laptoptracking.com`
- Password: `Admin@123`

**Test User Account:**
- Email: `user@laptoptracking.com`
- Password: `User@123`

## Setup Instructions

1. **Run the migrations:**
   ```bash
   php artisan migrate
   ```

2. **Seed the database with default users:**
   ```bash
   php artisan db:seed
   ```

3. **Clear cache (if needed):**
   ```bash
   php artisan config:clear
   php artisan cache:clear
   php artisan route:clear
   ```

## Routes

### Admin Routes (Protected by 'admin' middleware)
- `GET /users` - List all users
- `GET /users/create` - Show create user form
- `POST /users` - Store new user
- `GET /users/{user}` - Show user details and activity logs
- `GET /users/{user}/edit` - Show edit user form
- `PUT /users/{user}` - Update user
- `DELETE /users/{user}` - Delete user
- `GET /user-activities` - View all login activities

## Navigation

Admin users will see a "Users" menu item in the navigation bar. Regular users will not see this menu item.

## User Management Pages

### 1. Users List (`/users`)
- Displays all users with their roles, status, and last login
- Shows login count for each user
- Provides actions: View, Edit, Delete
- Paginated view

### 2. Create User (`/users/create`)
- Form to create new users
- Fields: Name, Email, Password, Role, Active status
- Password must be at least 8 characters

### 3. Edit User (`/users/{id}/edit`)
- Update user information
- Change password (optional)
- Toggle active status
- Change user role

### 4. User Details (`/users/{id}`)
- User information summary
- Login activity history with:
  - Date and time
  - IP address
  - Browser/device info
- Paginated activity log

### 5. All Login Activities (`/user-activities`)
- System-wide login activity log
- Shows all users' login activities
- Displays statistics:
  - Total login activities
  - Active users today
  - Total logins today
- Filterable and paginated

## Permissions

### Admin Can:
- View all users
- Create new users
- Edit any user (including role changes)
- Delete users (except themselves)
- View all login activities
- Activate/deactivate user accounts

### Regular User Can:
- Access dashboard and tracking features
- Update their own profile
- View their own information

## Security Considerations

1. **Inactive Users**: Users with `is_active = false` are automatically logged out when they attempt to log in
2. **Admin Protection**: Admins cannot delete their own accounts
3. **Password Security**: Passwords are hashed using Laravel's bcrypt
4. **Middleware Protection**: User management routes are protected by the `admin` middleware
5. **CSRF Protection**: All forms include CSRF tokens
6. **Route Protection**: Only authenticated admins can access user management features

## Middleware

### EnsureUserIsAdmin
Located at: `app/Http/Middleware/EnsureUserIsAdmin.php`

This middleware:
- Checks if user is authenticated
- Verifies user has admin role
- Returns 403 error for non-admin users

Usage in routes:
```php
Route::middleware(['admin'])->group(function () {
    // Admin-only routes
});
```

## Event Listeners

### LogUserLogin
Located at: `app/Listeners/LogUserLogin.php`

Listens to: `Illuminate\Auth\Events\Login`

Actions:
1. Checks if user is active (logs out if inactive)
2. Updates user's `last_login_at` timestamp
3. Creates activity log entry with IP and user agent

## Models

### UserActivityLog
Located at: `app/Models/UserActivityLog.php`

Relationships:
- `belongsTo(User::class)` - The user who performed the activity

### User (Updated)
Located at: `app/Models/User.php`

New Methods:
- `isAdmin()` - Returns boolean if user is admin

New Relationships:
- `hasMany(UserActivityLog::class)` - User's activity logs

## Views

All user management views are located in:
```
resources/views/users/
├── index.blade.php      # List all users
├── create.blade.php     # Create new user
├── edit.blade.php       # Edit user
├── show.blade.php       # User details & activity
└── activities.blade.php # All login activities
```

## Troubleshooting

### Issue: Cannot access user management pages
**Solution**: Make sure you're logged in as an admin user. Check your user's role in the database.

### Issue: Public registration is still showing
**Solution**: Clear your config cache: `php artisan config:clear`

### Issue: Login activities not being logged
**Solution**: 
1. Verify the event listener is registered in `AppServiceProvider`
2. Check database connection
3. Ensure `user_activity_logs` table exists

### Issue: "Column not found" error
**Solution**: Run migrations: `php artisan migrate`

## Future Enhancements

Potential features to add:
- Export user list to CSV/PDF
- Email notifications to new users
- Password reset functionality for admins
- Activity log filters (by date, user, IP)
- User sessions management
- Failed login attempt tracking
- Two-factor authentication enforcement by admin
- User groups/departments
- Bulk user import from CSV

## Maintenance

### Regular Tasks
1. Monitor login activities regularly
2. Review inactive users
3. Audit admin access logs
4. Update default admin password after first login
5. Backup user activity logs periodically

## Support

For issues or questions regarding the user management system, please contact your system administrator.

---

**Last Updated**: November 15, 2025
**Version**: 1.0.0
